# Copyright (C) 2000 by Dr. Dieter Maurer <dieter@handshake.de>
# D-66386 St. Ingbert, Eichendorffstr. 23, Germany
#
#			All Rights Reserved
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted,
# provided that the above copyright notice and this permission
# notice appear in all copies, modified copies and in
# supporting documentation.
# 
# Dieter Maurer DISCLAIMS ALL WARRANTIES WITH
# REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL Dieter Maurer
# BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
# DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
# PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
# TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
# PERFORMANCE OF THIS SOFTWARE.
"""CDROM ioctl adapter for Linux 2."""

from StructClass import genStructClass
import os, fcntl, FCNTL


# structures from "linux/cdrom.h"
cdrom_blk= genStructClass('cdrom_blk', 'Istart','Hlen',)

cdrom_msf= genStructClass('cdrom_msf',
	       'Bcdmsf_min0','Bcdmsf_sec0','Bcdmsf_frame0',
	       'Bcdmsf_min1','Bcdmsf_sec1','Bcdmsf_frame1',)

cdrom_ti= genStructClass('cdrom_ti',
	       'Bcdti_trk0','Bcdti_ind0','Bcdti_trk1','Bcdti_ind1',)

cdrom_tochdr= genStructClass('cdrom_tochdr',
	       'Bcdth_trk0','Bcdth_trk1',)

_cdrom_msf0= genStructClass('_cdrom_msf0',
	       'Bminute', 'Bsecond', 'Bframe',)

class cdrom_msf0(_cdrom_msf0):
  def __add__(self,msf):
    """add 2 msf0 to give an msf."""
    if not isinstance(msf,cdrom_msf0):
      raise TypeError,"msf must be an cdrom_msf0"
    return cdrom_msf(self.minute,self.second,self.frame,
		     msf.minute,msf.second,msf.frame)

_cdrom_tocentry= genStructClass('_cdrom_tocentry',
	       'Bcdte_track','Bcdte_adrctrl','Bcdte_format',
	       'Icdte_addr','Bcdte_datamode')

class cdrom_tocentry(_cdrom_tocentry):
  def usesLBAFormat(self): return self.cdte_adrctrl & 0x1
  def getAddr(self):
    cdte_addr= self.cdte_addr
    if self.usesLBAFormat(): return cdte_addr
    else: return bin2msf(cdte_addr)
  def getLBA(self):
    a= self.getAddr()
    if self.usesLBAFormat(): return a
    return msf2lba(a)
  def getMSF(self):
    a= self.getAddr()
    if self.usesLBAFormat(): return lba2msf(a)
    return a
  def isDataTrack(self): return cdte_adrctrl & (0x4 << 4)


cdrom_volctrl= genStructClass('cdrom_volctrl',
	       'Bchannel0','Bchannel1','Bchannel2','Bchannel3',)

_cdrom_subchnl= genStructClass('_cdrom_subchnl',
	        'Bcdsc_format', 'Bcdsc_audiostatus', 'Bcdsc_adrctrl',
		'Bcdsc_trk', 'Bcdsc_ind',
	        'Icdsc_absaddr', 'Icdsc_reladdr')

class cdrom_subchnl(_cdrom_subchnl):
  def usesLBAFormat(self): return self.cdsc_adrctrl & (0x1 << 4)
  def getAddr(self,rel=0):
    if rel: addr= self.cdsc_reladdr
    else: addr= self.cdsc_absaddr
    if self.usesLBAFormat(): return _sign(rel,addr)
    else: return bin2msf(addr)


class CDRom:
  def __init__(self,device):
    self.fd= os.open(device,FCNTL.O_RDONLY)
  def __del__(self): os.close(self.fd)
  #
  def pause(self): fcntl.ioctl(self.fd,0x5301)
  def resume(self): fcntl.ioctl(self.fd,0x5302)
  def playMSF(self,*msf):
    msf= _construct(cdrom_msf,msf)
    fcntl.ioctl(self.fd,0x5303,msf.toString())
  def playTI(self,*ti):
    ti= _construct(cdrom_ti,ti)
    fcntl.ioctl(self.fd,0x5304,ti.toString())
  def playBLK(self,*blk):
    blk= _construct(cdrom_blk,blk)
    fcntl.ioctl(self.fd,0x5317,blk.toString())
  def playFRAMES(self,startframe,no):
    self.playMSF(lba2msf(startframe) + lba2msf(startframe+no))
  #
  def readTocHdr(self):
    return cdrom_tochdr(fcntl.ioctl(self.fd,
				    0x5305,
				    cdrom_tochdr(0,0).toString())
			)
  def readTocEntry(self,i):
    return cdrom_tocentry(fcntl.ioctl(self.fd,
				      0x5306,
				      cdrom_tocentry(i,0,0,0,0).toString())
			  )
  def readToc(self):
    l= []
    hdr= self.readTocHdr()
    for i in range(hdr.cdth_trk0,hdr.cdth_trk1+1):
      l.append(self.readTocEntry(i))
    l.append(self.readTocEntry(0xaa))
    for i in range(1,len(l)):
      l[i-1].duration= l[i].getLBA() - l[i-1].getLBA()
    return l[:-1]
  #
  def stop(self):
    """stop motor."""
    fcntl.ioctl(self.fd,0x5307)
  def start(self):
    """start motor."""
    fcntl.ioctl(self.fd,0x5308)
  def eject(self):
    fcntl.ioctl(self.fd,0x5309)
  def reset(self):
    fcntl.ioctl(self.fd,0x5312)
  def closeTray(self):
    fcntl.ioctl(self.fd,0x5319)
  #
  def setVolume(self,*vol):
    vol= _construct(cdrom_volctrl,vol)
    fcntl.ioctl(self.fd,0x530a,vol.toString())
  def getVolume(self):
    return cdrom_volctrl(fcntl.ioctl(self.fd,
				     0x5313,
				     cdrom_volctrl(0,0,0,0).toString())
			 )
  def getSubchnl(self):
    return cdrom_subchnl(fcntl.ioctl(self.fd,
				     0x530b,
				     cdrom_subchnl(0,0,0,0,0,0,0).toString())
			 )
  

def lba2msf(a):
  """convert lba to msf."""
  f= a%75; a= a/75
  s= a%60; a= a/60
  m= a
  return cdrom_msf0(m,s,f)

def msf2lba(*m):
  """convert msf to lba."""
  m= _construct(cdrom_msf0,m)
  return (m.minute*60 + m.second)*75 + m.frame

def _construct(cl,args):
  if len(args) == 1 and isinstance(args[0],cl): return args[0]
  return apply(cl,args)

def bin2msf(bin):
  return cdrom_msf0(bin & 0xff, (bin >> 8) & 0xff, (bin >> 16) & 0xff)

def _sign(rel,val):
  if rel and val > (1 << 31): return int(val - (1L << 32))
  return int(val)
