# $Id: pattern.py,v 1.4 1999/04/11 16:09:15 dieter Exp dieter $
# Copyright (C) 1998-1999 by Dr. Dieter Maurer <dieter@handshake.de>
# D-66386 St. Ingbert, Eichendorffstr. 23, Germany
#
#			All Rights Reserved
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted,
# provided that the above copyright notice and this permission
# notice appear in all copies, modified copies and in
# supporting documentation.
# 
# Dieter Maurer DISCLAIMS ALL WARRANTIES WITH
# REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL Dieter Maurer
# BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
# DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
# PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
# TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
# PERFORMANCE OF THIS SOFTWARE.
"""XSL-Pattern

XSL-Pattern is an implementation of the pattern subset of
the XSL working draft 16-December-1998:
	URL:http://www.w3.org/TR/1998/WD-xsl-19981216

Patterns are used extensively in the XSL transformation
language and its control structures.
They can be used outside XSL, too, for e.g. querying/selecting
parts of SGML/XML documents.

XSL-Pattern is laid on top of PyDom, the DOM implementation
of Pythons XML special interest group:
	URL:http://www.python.org/sigs/xml-sig/files/xml-0.5.1.tgz
	URL:http://www.python.org/sigs/xml-sig/files/xml-0.5.1.zip

There are some deviations from the above XSL specification:

 * XSL requires namespace handling. This is not (yet) observed,
   because PyDom does not yet support namespaces.
   
 * 'id' references require the document to contain a method
   'getIdMap' returning a dictionary mapping id names to nodes.

   Such documents can be created with 'IdDecl' factories.
   They are constructed from an 'id' list and an optional
   element-id map, specifying which attributes are used
   as ids. Applied to a document, they add to it the
   required method.
   
 * CDATA sections and entity references are not yet handled correctly.
   Most parsers resolve CDATA sections and entity references
   into normal text nodes (if possible and not told otherwise).
   In the resulting DOM tree, there may be adjacent text nodes
   which is forbidden by the XSL specification.
   The function 'normalize' merges adjacent text nodes
   if applied to document.

 * The current version works with the ISO Latin 1 subset
   of Unicode rather than Unicode itself, violating the XML
   specification.
   The reason is that Python does not yet have full Unicode support.
"""

__version__= '0.3'

from PyBison.bisonparser import BisonParser
import patternparser
import re
from string import upper, replace, join
from xml.dom import core
from dmutil.visitor import Visitor


##############################################################################
## XML whitespace
WHITESPACE=r'[\x20\x9\xd\xa]'
_re_spaces= re.compile(WHITESPACE+'+')


##############################################################################
## XML character classes stripped down to 8 bit
# Letter
LETTER= '[\x41-\x5a\x61-\x7a\xc0-\xd6\xd8-\xf6\xf8-\xff]'
# XML NCNameChar (Namespace Constraint Name Char)
NCNAMECHAR= '(?:'+LETTER+'|[-\\d._\xb7])'
# XML NCName (Namespace Contraint Name)
NCNAME= '(?:(?:'+LETTER+'|_)'+NCNAMECHAR+'*)'


##############################################################################
## Error class


class Error(Exception):
  """General Errors"""

class SyntaxError(Error):
  """Syntax Errors"""




##############################################################################
## PatternParser

def _dictify(names):
  """dictionary with *names* as keys and corresponding tokens as value."""
  d= {}
  for name in names:
    d[name]= getattr(patternparser,upper(replace(name,'-','_')))
  return d


class _Parser(BisonParser):
  """Parser based on PyBisons 'BisonParser'.

  PyBison is a Python backend to the 'bison' parser
  generator. PyBison was developped by Scott Hassan
  URL:mailto://hassan@cs.stanford.edu.
  It (probably) can be found at:
  URL:http://www-db.stanford.edu/~hassan/Python/pybison.tar.gz.
  """
  def __init__(self,factory):
    BisonParser.__init__(self,patternparser,factory)
  #
  def parse(self,patternstring):
    """parses *patternstring* into an XSLPattern."""
    self._init_tokenize(patternstring)
    p= self.yyparse(None)
    p.patternstring= patternstring
    return p
  #
  __call__= parse
  #
  # Tokenizer
  _match_space= re.compile(WHITESPACE+'*').match
  _match_followsParen= re.compile(WHITESPACE+r'*\(').match
  _match_Qname= re.compile('(%s:)?(%s)' % (NCNAME,NCNAME)).match
  _match_Literal= re.compile('\'[^\']*\'|\"[^\"]*\"').match
  _seps= '/@()|[]=.*'
  _opnames= ('and', 'or')
  _opinhibitor= '@/|(['
  _funnames= ('id', 'ancestor', 'ancestor-or-self','comment','pi',
	      'text','not','first-of-type','last-of-type','first-of-any',
	      'last-of-any')
  _Operators= _dictify(_opnames)
  _isOperatorName= _Operators.has_key
  _Functions= _dictify(_funnames)
  _isFunctionName= _Functions.has_key
  #
  def _init_tokenize(self,patternstring):
    self._string= patternstring
    self._strlen= len(patternstring)
    self._strpos= 0
    self._inhibit_op= 1
  #
  def _skip_blank(self):
    m= self._match_space(self._string,self._strpos)
    if m: self._strpos= m.end()
  #
  def tokenize(self):
    self._skip_blank()
    p= self._strpos; n= self._strlen; s= self._string
    if p >= n: return ()
    c= s[p]
    if c in self._seps:
      p= p+1
      if c == '.' and p < n and s[p] == '.':
	self._inhibit_op= 0
	p= p+1
	tok= patternparser.PARENT
      else:
	self._inhibit_op= c in self._opinhibitor
	tok= ord(c)
    else:
      # it must be a name or a literal
      m= self._match_Qname(s,p)
      if m:
	# a name
	name= m.group(); p= m.end()
	if self._isOperatorName(name) and not self._inhibit_op:
	  tok= self._Operators[name]
	elif self._isFunctionName(name) \
	     and self._match_followsParen(s,p):
	  tok= self._Functions[name]
	else: tok= (patternparser.QNAME,name)
      else:
	m= self._match_Literal(s,p)
	if not m: self._syntax_error('unrecognized token')
	p= m.end()
	tok= (patternparser.LITERAL,m.group()[1:-1])
      self._inhibit_op= 0
    self._strpos= p
    return (tok,)
  #
  def _syntax_error(self,text):
    raise SyntaxError(text,self._string,self._strpos)
  #
  def yyerrlab1(self):
    self._syntax_error('invalid pattern syntax')
  


##############################################################################
## Pattern
class Pattern:
  """XSL-Pattern"""
  def __init__(self,pathlist):
    self._pathlist= pathlist
  #
  def select(self,node):
    """the list of nodes (in document order) selected by *self* starting from *node*.
    """
    s= _Selector(self._treePaths(),self._nonTreePaths())
    return s.getSelectedNodes(node)
  #
  def checkNonempty(self,node,value=None):
    """returns 'true' iff 'self.select(node)' is not empty."""
    s= _Checker(self._treePaths(),self._nonTreePaths(),value)
    return s.found(node)
  #
  def match(self,node):
    """returns 'true', if *node* is matched by *self*."""
    s= _Matcher()
    return s.matched(node,self._pathlist)
  #
  def _treePaths(self):
    try: return self._treepaths
    except AttributeError:
      self._splitPaths()
      return self._treepaths
  #
  def _nonTreePaths(self):
    try: return self._nontreepaths
    except AttributeError:
      self._splitPaths()
      return self._nontreepaths
  #
  def _splitPaths(self):
    tp= []; ntp= []
    for p in self._pathlist:
      n= p[0]
      if isinstance(n,Identity): tp.append(p)
      elif isinstance(n,OtherNode): ntp.append(p)
      else: tp.append([IDENTITY] + p)
    self._treepaths= tp
    self._nontreepaths= ntp


#########################################################################
# Navigators
#   the components of PathExpr are called navigators
#   they are: 
#     ROOT		the document root
#     DESCENDANTS	apply remaining list to all descendants
#     FilterExpr
DESCENDANTS= -1

class NodeExpr:
  """a (abstract) node expression -- possibly filtered."""
  _filter= None
  #
  def accept(self,node):
    o= self._accept(node)
    return o and (not self._filter or self._filter(node))
  #
  def addFilter(self,filter): self._filter= filter

class SubNode(NodeExpr):
  """a subnode expression."""
  def __init__(self,type,name= None):
    self._type= type; self._name= name
  #
  def _accept(self,node):
    return node.nodeType == self._type \
	   and (not self._name or node.nodeName == self._name)

class Element(SubNode):
  def __init__(self,name=None):
    SubNode.__init__(self,core.ELEMENT,name)

class Attribute(SubNode):
  def __init__(self,name=None):
    SubNode.__init__(self,core.ATTRIBUTE,name)


class OtherNode(NodeExpr):
  """they have an additional 'select' method."""
  def select(self,node):
    nl= self._select(node)
    if self._filter: return filter(self._filter,nl)
    return nl
  def rselect(self,node):
    """reverse select."""
    return (not self._filter or self._filter(node)) and self._rselect(node) or []

class Root(OtherNode):
  def _select(self,node): return [node.ownerDocument or node]
  def _accept(self,node): return node.nodeType == core.DOCUMENT

ROOT= Root()


class Id(OtherNode):
  """an ID selector.

  We require the DOM document to contain a method getIdMap returning
  a dictionary mapping id names to nodes.
  """
  def __init__(self,idref):
    self.idref= idref
  #
  def _select(self,node):
    idref= self.idref
    if isinstance(idref,Pattern): l= map(getNodeValue, idref.select(node))
    else: l= [idref]
    d= {}; idmap= (node.ownerDocument or node).getIdMap()
    for idrefs in l:
      for idref in _re_spaces.split(idrefs): d[idref]= None
    return filter(None,map(idmap.get,d.keys()))
  #
  def _accept(self,node):
    idref= self.idref
    if isinstance(idref,Pattern): return self._rselect(node)
    idmap= (node.ownerDocument or node).getIdMap()
    _node= node._node
    for id in _re_spaces.split(idref):
      n= idmap.get(id)
      if n and n._node == _node: return 1
    return 0
  #
  def accept(self,node):
    """redefinition, because '_accept' is so expensive."""
    return (not self._filter or self._filter(node)) and self._accept(node)
  #
  def _rselect(self,node):
    idref= self.idref
    doc= node.ownerDocument or node
    if not isinstance(idref,Pattern):
      return self._accept(node) and _IDWALKER.visit(doc) or []
    return _IDWALKER.visit(doc,
			   lambda n, sel=self._select, _node= node._node,
			     gn= lambda n: n._node:
			     _node in map(gn, sel(n)))

class _IdWalker(Visitor):
  ProcessAttributes=1
  def __init__(self):
    Visitor.__init__(self,self._check,)
  def visit(self,doc,checker= None):
    self._nodes= []; self._checker= checker
    self._visit(doc)
    return self._nodes
  def _check(self,node):
    checker= self._checker
    if checker is None or checker(node): self._nodes.append(node)

_IDWALKER= _IdWalker()

class Ancestor(OtherNode):
  def __init__(self,pattern,allow_self):
    self.pattern= pattern
    self.allow_self= allow_self
  #
  def _select(self,node):
    s= self.allow_self and node or node.parentNode
    p= self.pattern
    while s:
      if p.match(s): return [s]
      s= s.parentNode
    return []
  #
  def _accept(self,node):
    return self.pattern.match(node) and \
	   (self.allow_self or node.childNodes)
  #
  def _rselect(self,node):
    return _ANCWALKER.visit(node,self.pattern.match,self.allow_self)

class _AncWalker(Visitor):
  def __init__(self):
    Visitor.__init__(self,self._handleNode, self._handleDescendants)
  def visit(self,node,match,allow_self):
    self._node= node; self._match= match; self.allow_self= allow_self
    self._nodes= []
    self._visit(node)
    return self._nodes
  def _handleNode(self,node):
    if node is not self._node: return
    if self.allow_self: self._nodes.append(node)
    return 0
  def _handleDescendants(self,node):
    if self._match(node):
      if not self.allow_self: self._nodes.append(node)
      return 1
    self._nodes.append(node)

_ANCWALKER= _AncWalker()


class Identity(OtherNode):
  def _select(self,node): return [node]
  def _accept(self,node): return 1
  _rselect= _select

IDENTITY= Identity()


class Parent(OtherNode):
  def _select(self,node):
    p= node.parentNode
    return p and [p] or []
  #
  def _accept(self,node):
    return node.childNodes
  #
  def _rselect(self,node):
    return node.childNodes or []


#############################################################################
## Boolean Expressions
##   they can be applied to a node and return 'true' or 'false'

class BinBooleanExpr:
  def __init__(self,be):
    self._list= [be]
  def add(self,be):
    self._list.insert(0,be)

class Or(BinBooleanExpr):
  def __call__(self,node):
    for be in self._list:
      if be(node): return 1
    return 0

class And(BinBooleanExpr):
  def __call__(self,node):
    for be in self._list:
      if not be(node): return 0
    return 1


class Not:
  def __init__(self,be):
    self.be= be
  #
  def __call__(self,node): return not self.be(node)


class PosChecker:
  def __init__(self,dir,ty):
    self.dir= dir
    self.ty= ty
  #
  def __call__(self,node):
    if node.nodeType != core.ELEMENT: return 0
    if self.dir == -1: forward= core.Node.get_previousSibling
    else: forward= core.Node.get_nextSibling
    n= node; ty= self.ty
    while 1:
      n= forward(n)
      if n is None: return 1
      if n.nodeType == core.ELEMENT \
	 and (not ty or n.nodeName == node.nodeName): return 0


class Tester:
  def __init__(self,pattern,value=None):
    self.pattern= pattern
    self.value= value
  #
  def __call__(self,node): return self.pattern.checkNonempty(node,self.value)



##############################################################################
## Selecting Classes

class _Descender:
  """auxiliary class to evaluate 'select' and 'checkNonempty'."""
  _finished= 0
  def __init__(self,treepaths,nontreepaths):
    self.treepaths= treepaths
    self.nontreepaths= nontreepaths
  #
  def _walk(self,node,treepaths):
    """evaluate *treepaths* at node."""
    # check whether *node* is selected
    for p in treepaths:
      if len(p) == 1:
	ne= p[0]
	if ne.accept(node) and self._accept(node): break
    if self._finished: return
    # now determine the relevant set of subpath
    # and classify, whether they must be applies to attributes,
    # children or other nodes
    ap= []; cp= []; op= [];
    for p in treepaths:
      if len(p) == 1: continue
      ne= p[0]
      if ne is not DESCENDANTS:
	if not ne.accept(node): continue
	else: p= p[1:]; ne= p[0]
      if ne is DESCENDANTS:
	cp.append(p)
	p= p[1:]; ne= p[0]
      if isinstance(ne,Attribute): ap.append(p)
      elif isinstance(ne,OtherNode): op.append(p)
        # NOTE: if Python had "goto", if probably would have
	#       optimized 'isinstance(p,IDENTITY)' here.
      else: cp.append(p)
    # other nodes
    for o in op:
      self.unorderedSelections.append((o,node))
    if ap:
      # walk attributes
      attrs= node.attributes
      if attrs:
	for a in attrs.values():
	  self._walk(a,ap)
	  if self._finished: return
    if cp:
      # walk children
      for c in node.childNodes:
	self._walk(c,cp)
	if self._finished: return
  #
  def _search(self,node):
    nontreepaths= self.nontreepaths
    treepaths= self.treepaths
    self.unorderedSelections= s= map(None,nontreepaths,[node] * len(nontreepaths))
    if treepaths:
      self._walk(node,treepaths)
      if self._finished: return
      self._finish_walk()
    while s:
      (p,n)= s[0]; del s[0]
      ne= p[0]; ps= [[IDENTITY] + p[1:]]
      for n in ne.select(n):
	self._walk(n,ps)
	if self._finished: return
	self._finish_walk()
  #
  def _finish_walk(self): pass

class _Selector(_Descender):
  def getSelectedNodes(self,node):
    self.nodeseqlist= nsl= []
    self.selectednodes= []
    self._search(node)
    if not nsl: return []
    if len(nsl) == 1: return nsl[0]
    return _sortDocumentOrder(nsl)
  #
  def _finish_walk(self):
    if self.selectednodes:
      self.nodeseqlist.append(self.selectednodes)
      self.selectednodes= []
  #
  def _accept(self,node):
    self.selectednodes.append(node)
    return 1

class _Checker(_Descender):
  def __init__(self,treepaths,nontreepaths,value):
    self._value= value
    _Descender.__init__(self,treepaths,nontreepaths)
  #
  def found(self,node):
    self._search(node)
    return self._finished
  #
  def _accept(self,node):
    accepted= self._value is None or checkNodeValue(node,self._value)
    if accepted: self._finished= 1
    return accepted


def getNodeValue(node):
  """return the value of *node* -- as defined by XSL."""
  v= node.nodeValue
  if v is not None: return v
  return _valueVisitor.visit(node)

def checkNodeValue(node,value):
  """check whether *node* has value *value*."""
  v= node.nodeValue
  if v is not None: return v == value
  return _valueChecker.visit(node,value)

class _ValueVisitor(Visitor):
  def __init__(self):
    Visitor.__init__(self,self._classify)
  #
  def visit(self,node):
    """the value of *node*."""
    self.valfragments=[]
    self._visit(node)
    return join(self.valfragments,'')
  #
  def _classify(node):
    nt= node.nodeType
    if nt in (core.ELEMENT, core.DOCUMENT): return None
    if nt == core.TEXT: self.valfragments.append(node.nodeValue)
    return 1

_valueVisitor= _ValueVisitor()



class _ValueChecker(Visitor):
  def __init__(self):
    Visitor.__init__(self,self._classify)
  #
  def visit(self,node,value):
    """check the value of *node* against *value*."""
    self._value= value; self._vallen= len(value)
    self._valpref= ''; self._preflen= 0
    self._visit(node)
    return self._valpref == self._value
  #
  def _classify(self,node):
    if self._preflen > self._vallen: return 1
    nt= node.nodeType
    if nt in (core.ELEMENT, core.DOCUMENT): return None
    if nt == core.TEXT:
      v= node.nodeValue
      self._valpref= self._valpref + v
      self._preflen= self._preflen + len(v)
    return 1

_valueChecker= _ValueChecker()
    

class _DfsNumberer(Visitor):
  ProcessAttributes= 1
  def __init__(self):
    Visitor.__init__(self,self._numberer)
  def visit(self,doc):
    """dictionary mapping nodes to dfs number."""
    self._number= 0
    self._dfs= {}
    self._visit(doc)
    return self._dfs
  def _numberer(self,node):
    self._dfs[node._node]= self._number
    self._number= self._number + 1

dfsNumberer= _DfsNumberer()

def _sortDocumentOrder(nodeseqlist):
  """merge to nodesequences into a single nodesequence, ordered in document order.

  *nodeseqlist* is a list of nonempty sequences sorted in document order
  with at least 2 elements.
  """
  doc= nodeseqlist[0][0]
  doc= doc.ownerDocument or doc
  dfs= dfsNumberer.visit(doc)
  n= len(nodeseqlist); wl= [None] * n; i= 0
  for ns in nodeseqlist:
    node= ns[0]; wl[i]= (dfs[node._node],ns,node); i= i+1
  r= []
  while n > 1:
    wl.sort(); newsort= 0
    v1= wl[1][0]
    while 1:
      (v0,nl0,n0)= wl[0]
      if v0 > v1: break
      r.append(n0)
      if v0 == v1:
	newsort= 1
	j= 1
	while j < n:
	  wlj= wl[j]
	  if v0 == wlj[0]:
	    nl= wlj[1]
	    if len(nl) == 1:
	      n= n-1; j= j-1
	      del wl[j]
	    else:
	      del nl[0]
	      node= nl[0]
	      wl[j]= (dfs[node._node],nl,node)
	  else: break
	  j= j+1
      if len(nl0) == 1:
	n= n-1; del wl[0]
	break
      del nl0[0]
      node= nl0[0]
      wl[0]= (dfs[node._node],nl0,node)
      if newsort: break
  return r + wl[0][1]


class _Matcher:
  """upward tree walker for matching.

  ATTENTION: complex 'ancestor' and 'id' matches can be speeded up.
  """
  def matched(self,node,pathlist):
    self._specials= specials= []
    if self._walk(node,pathlist): return 1

    while specials:
      (n,p)= specials[0]; del specials[0]
      ne= p[-1]; p= p[:-1]
      if not p:
	if ne.accept(n): return 1
	continue
      for n in ne.rselect(n):
	if self._walk(n,[p]): return 1
    return 0
  #
  def _walk(self,node,pathlist):
    specials= self._specials
    while 1:
      for p in pathlist:
	if len(p) > 1: continue
	ne= p[0]
	if ne.accept(node): return 1
      pp= []
      for p in pathlist:
	if len(p) == 1: continue
	ne= p[-1]
	if ne is DESCENDANTS:
	  pp.append(p)
	  p= p[:-1]; ne= p[-1]
	if isinstance(ne,OtherNode): specials.append((node,p)); continue
	if ne.accept(node): 
	  if len(p) == 1: return 1
	  pp.append(p[:-1])
      node= node.parentNode
      if not node or not pp: return 0
      pathlist= pp
      


##############################################################################
## IdDecl

class IdDecl:
  """adding 'getIdMap' and 'flushIdMap' to a document."""
  def __init__(self,idlist=None,idmap=None):
    """creates a document factory providing a 'getIdMap' method to documents.

    If not 'None', *idlist* must be a list of attribute names.
    They are used as id attributes for element types not specified in
    *idmap*.

    If not 'None', *idmap* must be a dictionary mapping element types
    to attribute names. *idmap[e]=a* specifies that *a* is
    the id attribute for element type *e*.

    It is expensive to compute the IdMap of a document.
    Therefore, it is cached. If the document is changed,
    the idmap must be flushed by a call to the documents
    'flushIdMap' method.
    """
    self.generalIdPattern= idlist and Parser('@' + join(idlist,'| @')) or None
    if idmap:
      self.specificIdMap= nidmap= {}
      for k in idmap.keys(): nidmap[k]= idmap[k] and Parser('@' + idmap[k])
    else: self.specificIdMap= None
  #
  def __call__(self,document):
    """add methods 'getIdMap' and 'flushIdMap' to *document*."""
    if not isinstance(document,_Document):
      class _C(document.__class__,_Document): pass
      document.__class__= _C
    _Document.__init__(document,self.generalIdPattern,self.specificIdMap)

class _Document:
  """a document enhanced with 'getIdMap' and 'flushIdMap' methods."""
  __idmap= None
  def __init__(self,idpattern,idmap):
    self.__idpattern= idpattern; self.__specids= idmap
    self.__idmap= None
  #
  def __getattr__(self,attr): return getattr(self.__doc,attr)
  #
  def getIdMap(self):
    """return dictionary mapping ids to nodes (with this id)."""
    idmap= self.__idmap
    if idmap is not None: return idmap
    if self.__idpattern is None and self.__specids is None: idmap= {}
    else: idmap= _IDMAPPER.visit(self,self.__idpattern,self.__specids)
    self.__idmap= idmap
    return idmap
  #
  def flushIdMap(self): self.__idmap= None

class _IdMapper(Visitor):
  def __init__(self): Visitor.__init__(self,self._check)
  #
  def visit(self,doc,idpat,idmap):
    self._idpat= idpat; self._idmap= idmap
    self._dict= {}
    self._visit(doc)
    return self._dict
  #
  def _check(self,node):
    if node.nodeType == core.ELEMENT:
      idpat= self._idpat; idmap= self._idmap; et= node.nodeName
      if idmap and idmap.has_key(et):
        al= idmap[et]
	if al: al= al.select(node)
      elif idpat: al= idpat.select(node)
      else: return
      if not al: return
      if len(al) > 1:
	raise error,'element %s has %d id attributes' % (str(node),len(al))
      id= al[0].nodeValue
      d= self._dict
      if d.has_key(id):
	raise error,'id %s not unique' % id
      d[id]= node

_IDMAPPER= _IdMapper()


def normalize(doc):
  """normalize the document *doc*, i.e. merge adjacent text nodes."""
  doc.firstChild.normalize()


##############################################################################
## PatternFactory
class PatternFactory:
  """the standard factory used for pattern construction.

  Note that pattern matching may or may not work, if
  the factory is modified. This is because the classes
  used by the factory are not independent.
  It would be too difficult, to specify all dependencies, however.
  """
  def Pattern(self,pathexprlist): return Pattern(pathexprlist)
  def Element(self,qname=None): return Element(qname)
  def Attribute(self,qname=None): return Attribute(qname)
  def SubNode(self,domtype,name=None): return SubNode(domtype,name)
  def Id(self,idref): return Id(idref)
  def Ancestor(self,pattern,allow_self): return Ancestor(pattern,allow_self)
  def Identity(self): return Identity()
  def Parent(self): return Parent()
  def Or(self,node): return Or(node)
  def And(self,node): return And(node)
  def Not(self,booleanexpr): return Not(booleanexpr)
  def PosChecker(self,direction,type):
    """*direction*: -1 <-> first; 1 <-> last
       *type*:      1  <-> type;  0 <-> any

       e.g. *direction=-1, type=1* means FIRST_OF_TYPE
            *direction= 1, type=0* means LAST_OF_ANY
    """
    return PosChecker(direction,type)
  def Tester(self,pattern,value=None): return Tester(pattern,value)

# fixup
PatternFactory.ROOT= ROOT
PatternFactory.DESCENDANTS= DESCENDANTS



##############################################################################
## make Parser from factory
def makeParser(factory):
  """creates a parser from PatternFactory *factory*."""
  return _Parser(factory)


##############################################################################
## Standard Parser
Parser=makeParser(PatternFactory())
