/* $Id: _weakdict.c,v 1.3 1999/05/10 21:21:21 dieter Exp dieter $ */
/***********************************************************
Copyright 1991-1995 by Stichting Mathematisch Centrum, Amsterdam,
The Netherlands.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the names of Stichting Mathematisch
Centrum or CWI or Corporation for National Research Initiatives or
CNRI not be used in advertising or publicity pertaining to
distribution of the software without specific, written prior
permission.

While CWI is the initial source for this software, a modified version
is made available by the Corporation for National Research Initiatives
(CNRI) at the Internet address ftp://ftp.python.org.

STICHTING MATHEMATISCH CENTRUM AND CNRI DISCLAIM ALL WARRANTIES WITH
REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL STICHTING MATHEMATISCH
CENTRUM OR CNRI BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
PERFORMANCE OF THIS SOFTWARE.

******************************************************************/

/* Weak Dictionary object implementation by Dieter Maurer;
   derived from Python's dictionary implementation in "dictobject.c"
*/

#include "Python.h"


/*
 * MINSIZE is the minimum size of a dictionary.
 */

#define MINSIZE 4

/*
Table of irreducible polynomials to efficiently cycle through
GF(2^n)-{0}, 2<=n<=30.
*/
static long polys[] = {
	4 + 3,
	8 + 3,
	16 + 3,
	32 + 5,
	64 + 3,
	128 + 3,
	256 + 29,
	512 + 17,
	1024 + 9,
	2048 + 5,
	4096 + 83,
	8192 + 27,
	16384 + 43,
	32768 + 3,
	65536 + 45,
	131072 + 9,
	262144 + 39,
	524288 + 39,
	1048576 + 9,
	2097152 + 5,
	4194304 + 3,
	8388608 + 33,
	16777216 + 27,
	33554432 + 9,
	67108864 + 71,
	134217728 + 39,
	268435456 + 9,
	536870912 + 5,
	1073741824 + 83,
	0
};

/* Object used as dummy key to fill deleted entries */
static PyObject *dummy; /* Initialized by first call to newdictobject() */



typedef struct QueueElement {
  struct QueueElement	*fwd, *bwd; /* forward and backward pointers */
} QueueElement;

typedef struct {
  QueueElement	head;
  PyObject	*self;
} WeakValue;

typedef struct {
  QueueElement link;
  WeakValue    *value;

  struct WeakDict  *dict;	/* the dictionary */
  struct dictentry *pos;	/* position in dictionary */
} WeakValLink;

static void unlinkWeakValLink Py_PROTO((WeakValLink *));
static WeakValLink *linkWeakValue Py_PROTO((WeakValue *));


/*
Invariant for entries: when in use, de_key is
not NULL and not dummy; when not in use, de_value is NULL and de_key
is either NULL or dummy.  A dummy key value cannot be replaced by
NULL, since otherwise other keys may be lost.
*/
#define USED(ENTRY) ((ENTRY)->me_key && (ENTRY)->me_key != dummy)

typedef struct dictentry {
	long me_hash;
	PyObject 	*me_key;
	WeakValLink     *me_value; /* 0 for an occupied slot means "Py_None" */
#ifdef USE_CACHE_ALIGNED
	long		aligner;
#endif
} dictentry;

/*
To ensure the lookup algorithm terminates, the table size must be a
prime number and there must be at least one NULL key in the table.
The value ma_fill is the number of non-NULL keys; ma_used is the number
of non-NULL, non-dummy keys.
To avoid slowing down lookups on a near-full table, we resize the table
when it is more than half filled.
*/
typedef struct WeakDict{
	int ma_fill;
	int ma_used;
	int ma_size;
	int ma_poly;
	dictentry *ma_table;
} dictobject;

static PyObject * _createWeakDict Py_PROTO((PyObject *, PyObject *));

static PyObject *
_createWeakDict(self, args)
  PyObject	*self, *args;
{
	register dictobject *mp;

	if (! PyArg_ParseTuple(args,"")) return NULL;

	if (dummy == NULL) { /* Auto-initialize dummy */
		dummy = PyString_FromString("<dummy key>");
		if (dummy == NULL)
			return NULL;
	}
	mp = (dictobject *) malloc(sizeof(dictobject));
	if (mp == NULL) {
	  PyErr_NoMemory();
	  return NULL;
	}
	mp->ma_size = 0;
	mp->ma_poly = 0;
	mp->ma_table = NULL;
	mp->ma_fill = 0;
	mp->ma_used = 0;
	return Py_BuildValue("l",(long) mp);
}

/*
The basic lookup function used by all operations.
This is based on Algorithm D from Knuth Vol. 3, Sec. 6.4.
Open addressing is preferred over chaining since the link overhead for
chaining would be substantial (100% with typical malloc overhead).
However, instead of going through the table at constant steps, we cycle
through the values of GF(2^n)-{0}. This avoids modulo computations, being
much cheaper on RISC machines, without leading to clustering.

First a 32-bit hash value, 'sum', is computed from the key string.
The first character is added an extra time shifted by 8 to avoid hashing
single-character keys (often heavily used variables) too close together.
All arithmetic on sum should ignore overflow.

The initial probe index is then computed as sum mod the table size.
Subsequent probe indices use the values of x^i in GF(2^n) as an offset,
where x is a root. The initial value is derived from sum, too.

(This version is due to Reimer Behrends, some ideas are also due to
Jyrki Alakuijala and Vladimir Marangozov.)
*/
static dictentry *lookdict Py_PROTO((dictobject *, PyObject *, long));
static dictentry *
lookdict(mp, key, hash)
	dictobject *mp;
	PyObject *key;
	register long hash;
{
	register int i;
	register unsigned incr;
	register dictentry *freeslot;
	register unsigned int mask = mp->ma_size-1;
	dictentry *ep0 = mp->ma_table;
	register dictentry *ep;
	/* We must come up with (i, incr) such that 0 <= i < ma_size
	   and 0 < incr < ma_size and both are a function of hash */
	i = (~hash) & mask;
	/* We use ~hash instead of hash, as degenerate hash functions, such
	   as for ints <sigh>, can have lots of leading zeros. It's not
	   really a performance risk, but better safe than sorry. */
	ep = &ep0[i];
	if (ep->me_key == NULL)
		return ep;
	if (ep->me_key == dummy)
		freeslot = ep;
	else {
		if (ep->me_key == key ||
		 (ep->me_hash == hash &&
		  PyObject_Compare(ep->me_key, key) == 0))
		{
			return ep;
		}
		freeslot = NULL;
	}
	/* XXX What if PyObject_Compare returned an exception? */
	/* Derive incr from sum, just to make it more arbitrary. Note that
	   incr must not be 0, or we will get into an infinite loop.*/
	incr = (hash ^ ((unsigned long)hash >> 3)) & mask;
	if (!incr)
		incr = mask;
	else if (incr > mask) /* Cycle through GF(2^n)-{0} */
		incr ^= mp->ma_poly; /* This will implicitly clear the
					highest bit */
	for (;;) {
		ep = &ep0[(i+incr)&mask];
		if (ep->me_key == NULL) {
			if (freeslot != NULL)
				return freeslot;
			else
				return ep;
		}
		if (ep->me_key == dummy) {
			if (freeslot == NULL)
				freeslot = ep;
		}
		else if (ep->me_key == key ||
			 (ep->me_hash == hash &&
			  PyObject_Compare(ep->me_key, key) == 0)) {
			return ep;
		}
		/* XXX What if PyObject_Compare returned an exception? */
		/* Cycle through GF(2^n)-{0} */
		incr = incr << 1;
		if (incr > mask)
			incr ^= mp->ma_poly;
	}
}

/*
Internal routine to insert a new item into the table.
Used both by the internal resize routine and by the public insert routine.
Eats a reference to key.
*/
static void insertdict
	Py_PROTO((dictobject *, PyObject *, long, WeakValLink *));
static void
insertdict(mp, key, hash, value)
	register dictobject *mp;
	PyObject *key;
	long hash;
	WeakValLink *value;
{
	register dictentry *ep;
	ep = lookdict(mp, key, hash);
	if (USED(ep)) {
		unlinkWeakValLink(ep->me_value);
		Py_DECREF(key);
	}
	else {
		if (ep->me_key == NULL)
			mp->ma_fill++;
		else
			Py_DECREF(ep->me_key);
		ep->me_key = key;
		ep->me_hash = hash;
		mp->ma_used++;
	}
	ep->me_value = value;
	if (value) {
	  value->dict= mp;
	  value->pos = ep;
	}
}

/*
Restructure the table by allocating a new table and reinserting all
items again.  When entries have been deleted, the new table may
actually be smaller than the old one.
*/
static int dictresize Py_PROTO((dictobject *, int));
static int
dictresize(mp, minused)
	dictobject *mp;
	int minused;
{
	register int oldsize = mp->ma_size;
	register int newsize, newpoly;
	register dictentry *oldtable = mp->ma_table;
	register dictentry *newtable;
	register dictentry *ep;
	register int i;
	for (i = 0, newsize = MINSIZE; ; i++, newsize <<= 1) {
		if (i > sizeof(polys)/sizeof(polys[0])) {
			/* Ran out of polynomials */
			PyErr_NoMemory();
			return -1;
		}
		if (newsize > minused) {
			newpoly = polys[i];
			break;
		}
	}
	newtable = (dictentry *) calloc(sizeof(dictentry), newsize);
	if (newtable == NULL) {
		PyErr_NoMemory();
		return -1;
	}
	mp->ma_size = newsize;
	mp->ma_poly = newpoly;
	mp->ma_table = newtable;
	mp->ma_fill = 0;
	mp->ma_used = 0;

	/* Make two passes, so we can avoid decrefs
	   (and possible side effects) till the table is copied */
	for (i = 0, ep = oldtable; i < oldsize; i++, ep++) {
		if (USED(ep))
			insertdict(mp,ep->me_key,ep->me_hash,ep->me_value);
	}
	for (i = 0, ep = oldtable; i < oldsize; i++, ep++) {
		if (! USED(ep)) {
			Py_XDECREF(ep->me_key);
		}
	}

	PyMem_XDEL(oldtable);
	return 0;
}


static PyObject * _getWeakItem Py_PROTO((PyObject *, PyObject *));

static PyObject *
_getWeakItem(self, args)
  PyObject	*self, *args;
{
	long hash;
	unsigned long	   lop;
	dictobject  	   *op;
	dictentry   	   *de;
	WeakValLink	   *vallink;
	PyObject	   *key;
	PyObject	   *res;

	if (! PyArg_ParseTuple(args,"lO",&lop,&key)) return NULL;
	op= (dictobject *) lop;

	if ((op)->ma_table == NULL)
		goto empty;
#ifdef CACHE_HASH
	if (!PyString_Check(key) ||
	    (hash = ((PyStringObject *) key)->ob_shash) == -1)
#endif
	{
		hash = PyObject_Hash(key);
		if (hash == -1)
			return NULL;
	}
	de= lookdict((dictobject *)op, key, hash);
	if (USED(de)) {
	  vallink= de->me_value;
	  if (vallink) res= vallink->value->self;
	  else res= Py_None;
	  Py_INCREF(res);
	} else {
	empty:
	  PyErr_SetObject(PyExc_KeyError, key);
	  return NULL;
	}
	return res;
}


static PyObject * _setWeakItem Py_PROTO((PyObject *, PyObject *));

static PyObject *
_setWeakItem(self, args)
  PyObject	*self, *args;
{
	register dictobject *mp;
	register long hash;
	PyObject	*key, *oval;
	WeakValLink	*val;
	unsigned long	lmp;

	if (! PyArg_ParseTuple(args,"lOO",&lmp,&key,&oval)) return NULL;
	mp = (dictobject *)lmp;

#ifdef CACHE_HASH
	if (PyString_Check(key)) {
#ifdef INTERN_STRINGS
		if (((PyStringObject *)key)->ob_sinterned != NULL) {
			key = ((PyStringObject *)key)->ob_sinterned;
			hash = ((PyStringObject *)key)->ob_shash;
		}
		else
#endif
		{
			hash = ((PyStringObject *)key)->ob_shash;
			if (hash == -1)
				hash = PyObject_Hash(key);
		}
	}
	else
#endif
	{
		hash = PyObject_Hash(key);
		if (hash == -1)
			return NULL;
	}
	/* if fill >= 2/3 size, double in size */
	if (mp->ma_fill*3 >= mp->ma_size*2) {
		if (dictresize(mp, mp->ma_used*2) != 0) {
			if (mp->ma_fill+1 > mp->ma_size)
				return NULL;
		}
	}
	if (oval == Py_None) val= NULL;
	else {
	  val= linkWeakValue((WeakValue *) PyInt_AsLong(oval));
	  if (!val) return NULL;
	}
	Py_INCREF(key);
	insertdict(mp, key, hash, val);
	Py_INCREF(Py_None);
	return Py_None;
}


static PyObject * _delWeakItem Py_PROTO((PyObject *, PyObject *));

static PyObject *
_delWeakItem(self, args)
  PyObject	*self, *args;
{
	register dictobject *mp;
	unsigned long	    lmp;
	register long hash;
	register dictentry *ep;
	PyObject *key;

	if (! PyArg_ParseTuple(args,"lO",&lmp,&key)) return NULL;
	mp= (dictobject *) lmp;

#ifdef CACHE_HASH
	if (!PyString_Check(key) ||
	    (hash = ((PyStringObject *) key)->ob_shash) == -1)
#endif
	{
		hash = PyObject_Hash(key);
		if (hash == -1) goto empty;
	}
	if (mp->ma_table == NULL)
		goto empty;
	ep = lookdict(mp, key, hash);
	if (! USED(ep)) {
	empty:
		PyErr_SetObject(PyExc_KeyError, key);
		return NULL;
	}
	key = ep->me_key;
	Py_INCREF(dummy);
	ep->me_key = dummy;
	unlinkWeakValLink(ep->me_value);
	ep->me_value = NULL;
	mp->ma_used--;
	Py_DECREF(key); 
	Py_INCREF(Py_None);
	return Py_None;
}


static PyObject * _deleteWeakDict Py_PROTO((PyObject *, PyObject *));

static PyObject *
_deleteWeakDict(self, args)
  PyObject	*self, *args;
{
	int i, n;
	register dictentry *table;
	dictobject *mp;
	unsigned long lmp;

	if (! PyArg_ParseTuple(args,"l",&lmp)) return NULL;

	mp = (dictobject *)lmp;
	table = mp->ma_table;
	if (table == NULL) goto ret;
	n = mp->ma_size;
	mp->ma_size = mp->ma_used = mp->ma_fill = 0;
	mp->ma_table = NULL;
	for (i = 0; i < n; i++) {
		unlinkWeakValLink(table[i].me_value);
		Py_XDECREF(table[i].me_key);
	}
	PyMem_DEL(table);
ret:
	free(mp);
	Py_INCREF(Py_None);
	return Py_None;
}


static PyObject * _getWeakKeys Py_PROTO((PyObject *, PyObject *));

static PyObject *
_getWeakKeys(self, args)
	PyObject *self,*args;
{
	register dictobject *mp;
	register PyObject *v;
	register int i, j;
	unsigned long lmp;
	if (!PyArg_ParseTuple(args,"l",&lmp))
		return NULL;
	mp= (dictobject *) lmp;
	v = PyList_New(mp->ma_used);
	if (v == NULL)
		return NULL;
	for (i = 0, j = 0; i < mp->ma_size; i++) {
		if (USED(&mp->ma_table[i])) {
			PyObject *key = mp->ma_table[i].me_key;
			Py_INCREF(key);
			PyList_SetItem(v, j, key);
			j++;
		}
	}
	return v;
}


static PyObject * _getWeakDictLength Py_PROTO((PyObject *, PyObject *));

static PyObject *
_getWeakDictLength(self, args)
  PyObject	*self, *args;
{
	register dictobject *mp;
	unsigned long	    lmp;
	if (!PyArg_ParseTuple(args,"l",&lmp))
		return NULL;
	mp= (dictobject *) lmp;

	return Py_BuildValue("l",mp->ma_used);
}


static PyObject * _createWeakValue Py_PROTO((PyObject *, PyObject *));

static PyObject *
_createWeakValue(self, args)
  PyObject	*self, *args;
{
  PyObject *val;
  WeakValue *weakval;

  if (! PyArg_ParseTuple(args,"O",&val)) return NULL;
  weakval= (WeakValue *) malloc(sizeof(WeakValue));
  if (! weakval) {
    PyErr_NoMemory(); return NULL;
  }
  weakval->head.fwd= weakval->head.bwd= (QueueElement *)weakval;
  weakval->self= val; /* not incref'ed! */
  return Py_BuildValue("l",(unsigned long) weakval);
}


static PyObject * _deleteWeakValue Py_PROTO((PyObject *, PyObject *));

static PyObject *
_deleteWeakValue(self, args)
  PyObject	*self, *args;
{
  unsigned long	lweakval;
  WeakValue	*weakval;
  QueueElement  *qe;
  
  if (! PyArg_ParseTuple(args,"l",&lweakval)) return NULL;
  weakval= (WeakValue *) lweakval;
  
  while ((qe= weakval->head.fwd) != (QueueElement *)weakval) {
    WeakValLink	*wlv= (WeakValLink *) qe;
    dictentry   *pos= wlv->pos;
    PyObject	*key= pos->me_key;

    wlv->dict->ma_used--; /* maybe, we should shrink */
    unlinkWeakValLink(wlv);
    Py_INCREF(dummy); pos->me_key= dummy; pos->me_value= NULL;
    Py_DECREF(key);
  }

  free(weakval);

  Py_INCREF(Py_None);
  return Py_None;
}


static PyMethodDef moduleMethods[] = {
	{"_createWeakValue", _createWeakValue, 1, },
	{"_deleteWeakValue", _deleteWeakValue, 1, },
	{"_createWeakDict", _createWeakDict, 1, },
	{"_deleteWeakDict", _deleteWeakDict, 1, },
	{"_setWeakItem", _setWeakItem, 1, },
	{"_getWeakItem", _getWeakItem, 1, },
	{"_delWeakItem", _delWeakItem, 1, },
	{"_getWeakKeys", _getWeakKeys, 1, },
	{"_getWeakDictLength", _getWeakDictLength, 1, },
	{NULL,		NULL}		/* sentinel */
};


static void unlinkWeakValLink (weaklink)
  WeakValLink	*weaklink;
{
  if (weaklink) {
    weaklink->link.bwd->fwd= weaklink->link.fwd;
    weaklink->link.fwd->bwd= weaklink->link.bwd;
    free(weaklink);
  }
}

static WeakValLink *linkWeakValue (weakval)
  WeakValue	*weakval;
{
  WeakValLink	*wlv;

  wlv= (WeakValLink *) malloc(sizeof(WeakValLink));
  if (! wlv) {
    PyErr_NoMemory();
    return NULL;
  }
  wlv->value= weakval;
  wlv->link.fwd= weakval->head.fwd; wlv->link.bwd= (QueueElement *) weakval;
  weakval->head.fwd->bwd= (QueueElement *) wlv;
  weakval->head.fwd= (QueueElement *) wlv;
  return wlv;
}



/******************************************************************
# initialization
*/
void
init_weakdict() {

  Py_InitModule("_weakdict",moduleMethods);

  if (PyErr_Occurred ())
    Py_FatalError ("can't initialize module _weakdict");
}
