#	$Id: weakdict.py,v 1.2 1999/05/10 21:21:21 dieter Exp dieter $
# Copyright (C) 1998 by Dr. Dieter Maurer <dieter@handshake.de>
# D-66386 St. Ingbert, Eichendorffstr. 23, Germany
#
#			All Rights Reserved
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose and without fee is hereby granted,
# provided that the above copyright notice and this permission
# notice appear in all copies, modified copies and in
# supporting documentation.
# 
# Dieter Maurer DISCLAIMS ALL WARRANTIES WITH
# REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL Dieter Maurer
# BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
# DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
# PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
# TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
# PERFORMANCE OF THIS SOFTWARE.
"""\
WeakDict: addressing CPythons problems with cyclic references.

WeakDict (Weak Dictionaries) have been designed to address
CPythons problems with cyclic references (cyclic structures
cannot be garbaged collected by CPythons reference counting
scheme). More precisely, WeakDict's allow the realization
of weak references, references that are **NOT** counted in
the reference count and can therefor be used to build
cyclic structures without obstructing the reference counting
scheme.

WeakDict's are very similar to normal Python dictionaries,
with the following essential exceptions:

 - all values in a WeakDict must be instances of 'WeakValue'
   (or a derived class)

 - the reference to a value in a WeakDict is *NOT* counted
   in the reference count of the value.
   Thus, it does not prevent the value from being garbaged collected.

 - When a value is garbaged collected, the corresponding
   entry disappears from the WeakDict.

WeakDict's have a timing efficiency similar to normal
Python dicts. Deletion of WeakValue's can take linear
time in the number of uses in WeakDict's.

The implementation is mostly in Python with some functions
realized in 'weakdict.c'.
"""

# import "C" infrastructure
from _weakdict import _createWeakValue, _deleteWeakValue, \
                      _createWeakDict, _deleteWeakDict, \
		      _setWeakItem, _getWeakItem, _delWeakItem, \
		      _getWeakKeys, _getWeakDictLength

class WeakValue:
  """base class for values usable in 'WeakDict's."""
  __cid= None		# pointer to C extension structure
                        # tempering with it risks SIGSEGV
  def __init__(self):
    if self.__cid is not None: return # already initialized -- multiple inheritance
    self.__cid= _createWeakValue(self)
  #
  def __del__(self):
    cid= self.__cid
    if cid is not None:
      del self.__cid
      _deleteWeakValue(cid)
  #
  def _getCid(self):
    cid= self.__cid
    if cid is None: raise ValueError,'uninitialized WeakValue used'
    return cid


class WeakDict:
  """Dictionary of weak references."""
  __cid= None		# pointer to C extension structure
                        # tempering with it risks SIGSEGV
  def __init__(self):
    if self.__cid is not None: return # already initialized -- multiple inheritance
    self.__cid= _createWeakDict()
  #
  def __del__(self):
    cid= self.__cid
    if cid is not None:
      del self.__cid
      _deleteWeakDict(cid)
  #
  def _checkval(self,val):
    """*val* must a 'WeakValue'. Its Cid is returned."""
    if isinstance(val,WeakValue): return val._getCid()
    raise TypeError,'value for WeakDict.__setitem__ is not a WeakValue'
  # emulate Python's dict operations
  def __getitem__(self,key): return _getWeakItem(self.__cid,key)
  def __setitem__(self,key,val):
    """*val* must a 'WeakValue'."""
    _setWeakItem(self.__cid,key,self._checkval(val))
  def __delitem__(self,key): _delWeakItem(self.__cid,key)
  def has_key(self,key):
    try: self.__getitem__(key); return 1
    except KeyError: return 0
  def __len__(self): return _getWeakDictLength(self.__cid)
  def keys(self): return _getWeakKeys(self.__cid)
  def values(self): return map(self.__getitem__,self.keys())
  def items(self): return map(lambda v,gv=self.__getitem__: (v,gv(v)), self.keys())
  def get(self,key):
    try: return self.__getitem__(key)
    except KeyError: return None
  def clear(self):
    for k in self.keys(): self.__delitem__(k)
  def copy(self):
    d= WeakDict()
    for k in self.keys(): d[k]= self.__getitem__(k)
    return d
  def update(self,d):
    for k in d.keys(): self.__setitem__(k,d[k])
  #
  # a convenience function similar to 'get' -- remove an entry
  def rem(self,key):
    try: self.__delitem__(key)
    except KeyError: pass
